<?php
/**
* @package   com_zoo
* @author    Mischa http://www.omnislab.com
* @copyright Copyright (C) OmnisLAB 2014
* @version   1.2.0
* @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
*/

/*
	Class: ElementBackRelatedItems
		The back related items element class
*/
class ElementBackRelatedItems extends Element {

	protected $_related_items;

	/*
		Function: hasValue
			Checks if the element's value is set.

	   Parameters:
			$params - render parameter

		Returns:
			Boolean - true, on success
	*/
	public function hasValue($params = array()) {
		$items = $this->_getBackRelatedItems();
		return !empty($items);
	}

	/*
		Function: render
			Renders the element.

	   Parameters:
            $params - render parameter

		Returns:
			String - html
	*/
	public function render($params = array()) {

		// init vars
		$params   = $this->app->data->create($params);
		$renderer = $this->app->renderer->create('item')->addPath(array($this->app->path->path('component.site:'), $this->_item->getApplication()->getTemplate()->getPath()));
		$items    = $this->_orderItems($this->_getBackRelatedItems(), $params->get('order'));
	
		// create output
		$layout   = $params->get('layout');
		$output   = array();
		
		foreach ($items as $item) {
			$path   = 'item';
			$prefix = 'item.';
			$type   = $item->getType()->id;
			if ($renderer->pathExists($path.DIRECTORY_SEPARATOR.$type)) {
				$path   .= DIRECTORY_SEPARATOR.$type;
				$prefix .= $type.'.';
			}

			if (in_array($layout, $renderer->getLayouts($path))) {
				$output[] = $renderer->render($prefix.$layout, array('item' => $item));
			} elseif ($params->get('link_to_item', false) && $item->getState()) {
				$output[] = '<a href="'.$this->app->route->item($item).'" title="'.$item->name.'">'.$item->name.'</a>';
			} else {
				$output[] = $item->name;
			}
		}
		
		return $this->app->element->applySeparators($params->get('separated_by'), $output);
	}

	
	protected function _orderItems($items, $order) {

		// if string, try to convert ordering
		if (is_string($order)) {
			$order = $this->app->itemorder->convert($order);
		}

		$items = (array) $items;
		$order = (array) $order;
		$sorted = array();
		$reversed = false;

		// remove empty values
		$order = array_filter($order);

		// if random return immediately
		if (in_array('_random', $order)) {
			shuffle($items);
			return $items;
		}

		// get order dir
		if (($index = array_search('_reversed', $order)) !== false) {
			$reversed = true;
			unset($order[$index]);
		} else {
			$reversed = false;
		}

		// order by default
		if (empty($order)) {
			return $reversed ? array_reverse($items, true) : $items;
		}

		// if there is a none core element present, ordering will only take place for those elements
		if (count($order) > 1) {
			$order = array_filter($order, create_function('$a', 'return strpos($a, "_item") === false;'));
		}

		if (!empty($order)) {

			// get sorting values
			foreach ($items as $item) {
				foreach ($order as $identifier) {
					if ($element = $item->getElement($identifier)) {
						$sorted[$item->id] = strpos($identifier, '_item') === 0 ? $item->{str_replace('_item', '', $identifier)} : $element->getSearchData();
						break;
					}
				}
			}

			// do the actual sorting
			$reversed ? arsort($sorted) : asort($sorted);

			// fill the result array
			foreach (array_keys($sorted) as $id) {
				if (isset($items[$id])) {
					$sorted[$id] = $items[$id];
				}
			}

			// attach unsorted items
			$sorted += array_diff_key($items, $sorted);

		// no sort order provided
		} else {
			$sorted = $items;
		}

		return $sorted;
	}

	/*
	   Function: _getBackRelatedItems
	       fetches the related items of parent.

	   Returns:
	       array _related_items
	*/
	
	protected function _getBackRelatedItems() {

		if ($this->_related_items == null) {
		
			// init vars
			$this->_related_items = array();
			$related_items = array();
			
			$query = array('select' => 'a.*',
						'from' => ZOO_TABLE_ITEM.' AS a'
						.' LEFT JOIN '.ZOO_TABLE_TAG.' AS b ON a.id = b.item_id'
						.' LEFT JOIN '.ZOO_TABLE_CATEGORY_ITEM.' AS c ON a.id = c.item_id',
						'conditions' => $this->_getConditions());
						
			// get items
			$related_items = $this->app->table->item->all($query);
	
			foreach($related_items as $item) {
				if ($this->_isRelated($item)) {
					$this->_related_items[$item->id] = $item;
				}
			}
		}

		return $this->_related_items;
	}
	
	/*
	   Function: edit
	       Renders the edit form field.

	   Returns:
	       null
	*/
	public function edit() {
		return null;
	}

	/*
		Function: _isRelated
			check if item is related to parent
		
		Returns:
			bool
	*/
			
	private function _isRelated($item) {
		// init vars
		$parentid = $this->getItem()->id;
		
		foreach ($item->getElements() as $element) {
			if ($element->config->get('type') == 'relateditems') {
	
				$itemids = $element->get('item', array());
				foreach ($itemids as $key => $itemid) {
					if ($itemid ==  $parentid) {
						//unset($itemids[$key]);
						return true;
					}
				}
			}	
		}
		return false;
	}
	
	/*
		Function: _getConditions
			sets query filters
		
		Returns:
			Query String
	*/
	
	private function _getConditions() {
		// get database
		$db = $this->app->database;	
	
		// set base query options
		$heuristic = '": "'.$this->getItem()->id.'"'; // narrow down possible candidates
		$heuristic_s = '": '.$this->getItem()->id; // difference in submission form (id stored as number)
		$conditions = "INSTR (a.elements, '".$heuristic."') OR INSTR (a.elements, '".$heuristic_s."')AND a.application_id = ".$this->_item->getApplication()->id;
		
		// FILTERS

		// ... by types

		$option = $this->config->get('type_options', 0);
		
		if ($option == "2") {
			$conditions .= " AND a.type = '".$this->getItem()->getType()->id."'";
		
		} elseif (($types = $this->config->get('selectable_types')) && !empty($types)) {			
			$typeIds = array();
			foreach($types as $id) {
				if(!empty($id)) {
					$typeIds[] = $db->Quote($db->escape((string) $id));
				}
			}

			if(!empty($typeIds)) {
				$conditions .= " AND a.type".($option == "0" ? "" : " NOT")." IN (".implode(', ', $typeIds).")";
			}
		}

		// ... by categories
		
		$option = $this->config->get('category_options', 0);
		
		if ($option == "2") {
			if(($categories = $this->_item->getRelatedCategories(true)) && !empty($categories)) {
				$categoryIds = array();
				foreach($categories as $id) {
					if(!empty($id)) {
						$categoryIds[] = $db->Quote((int) $id);
					}
				}
			
				$conditions .= " AND c.category_id IN (".implode(', ', $categoryIds).")";
			}
		} elseif(($categories = $this->config->get('selectable_categories')) && !empty($categories)) {
			$categoryIds = array();
			foreach($categories as $id) {
				if(!empty($id)) {
					$categoryIds[] = $db->Quote((int) $id);
				}
			}
			$conditions .= " AND c.category_id".($option == "0" ? "" : " NOT")." IN (".implode(', ', $categoryIds).")";
		}
			
		// ... by tags
		
		$option = $this->config->get('tag_options', 0);
		
		if ($option == "2") {
			if(($tags = $this->_item->getTags()) && !empty($tags)) {
				$tagNames = array();
				foreach($tags as $name) {
					if(!empty($name)) {
						$tagNames[] = $db->Quote($db->escape((string) $name));
					}
				}
			
				$conditions .= " AND b.name IN (".implode(', ', $tagNames).")";
			}
		} elseif(($tags = $this->config->get('selectable_tags')) && !empty($tags)) {
			$tagNames = array();
			foreach($tags as $name) {
				if(!empty($name)) {
					$tagNames[] = $db->Quote($db->escape((string) $name));
				}
			}
			$conditions .= " AND b.name".($option == "0" ? "" : " NOT")." IN (".implode(', ', $tagNames).")";
		}
		
		// ... by selected authors
		
		if($this->config->get('author_options', 0) == "1") {
			$conditions .= " AND a.created_by = ".$db->Quote($db->escape((int) $this->getItem()->created_by));
			
		} elseif(($authors = $this->config->get('selectable_authors')) && !empty($authors)) {
			$authorIds = array();
			foreach($authors as $id) {
				if(!empty($id)) {
					$authorIds[] = $db->Quote((int) $id);
				}
			}
			if(!empty($authorIds)) {
				$conditions .= " AND a.created_by IN (".implode(', ', $authorIds).")";
			}
		}
		
		// ... expired
		if(!$this->config->get('expired', 0)) {
			// get dates and prepare
			$date = $this->app->date->create();
			$now  = $db->Quote($date->toSQL());
			$null = $db->Quote($db->getNullDate());
			$conditions .= " AND (a.publish_up = ".$null." OR a.publish_up <= ".$now.")" 
						." AND (a.publish_down = ".$null." OR a.publish_down >= ".$now.")"; 
		}
		
		// ... published
		if($this->config->get('published', 1)) {			
			$conditions .= " AND a.state = 1";
		}
		return $conditions;
	}

	/*
		Function: getConfigForm
			Get parameter form object to render input form.

		Returns:
			Parameter Object
	*/
	
	public function getConfigForm() {
		return parent::getConfigForm()->addElementPath(dirname(__FILE__));
	}

}